#!/usr/bin/python3

def read_xyz(path):
    with open(path, 'r') as f:
        lines = f.readlines()
    natoms = int(lines[0].strip())
    title = lines[1].rstrip("\n")
    coords = [line.rstrip("\n") for line in lines[2:]]
    if len(coords) != natoms:
        raise ValueError("The number of atoms is inconsistent with the first line")
    return natoms, title, coords

def write_xyz(path, natoms, title, coords):
    with open(path, 'w') as f:
        f.write(f"{natoms}\n")
        f.write(f"{title}\n")
        for line in coords:
            f.write(f"{line}\n")

def translate_atoms(coords, vector, start_idx, end_idx):
    dx, dy, dz = vector
    for i in range(start_idx, end_idx + 1):
        if i >= len(coords):
            break
        parts = coords[i].split()
        if len(parts) < 4:
            raise ValueError(f"Line {i+3} format error: {coords[i]}")
        sym, x, y, z = parts[0], float(parts[1]), float(parts[2]), float(parts[3])
        x += dx
        y += dy
        z += dz
        coords[i] = f"{sym:>2s} {x:15.8f} {y:15.8f} {z:15.8f}"
    return coords

if __name__ == "__main__":
    dx = (-0.198382)-1.086004
    dy = (0.676975)-(-1.277514)
    dz = (0.839378)-(-1.352322)
    r = (dx*dx+dy*dy+dz*dz)**0.5
    ex = dx/r
    ey = dy/r
    ez = dz/r
    in_file = "r0.xyz"
    for i in range(1,10):
        natoms, title, coords = read_xyz(in_file)
        dr = 0.5*i
        d = 3.21+dr
        vector = (ex*dr, ey*dr, ez*dr)
        coords = translate_atoms(coords, vector, start_idx=0, end_idx=9)
        out_file = f"r{i:d}.xyz"
        title = f"d = {d:.2f} Angstrom"
        write_xyz(out_file, natoms, title, coords)
        print(f"Write {out_file}.")
